<?php
use yii\helpers\Html;
use yii\widgets\ActiveForm;
?>

<!-- Основной контент -->
<div id="create-notification" class="page">
    <div class="flex justify-between items-center mb-8">
        <div>
            <h2 class="text-3xl font-bold text-gray-800">Создать уведомление</h2>
            <p class="text-gray-600 mt-2">Отправьте сообщение пользователям системы</p>
        </div>
        <?= Html::a('← Назад к списку', ['notifications'], [
            'class' => 'text-blue-600 hover:text-blue-800 flex items-center'
        ]) ?>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- Основная форма -->
        <div class="bg-white rounded-xl shadow-md p-6 lg:col-span-2">
            <div class="flex items-center mb-6">
                <div class="p-3 bg-blue-100 rounded-lg mr-4">
                    <span class="text-2xl text-blue-600">🔔</span>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800">Новое уведомление</h3>
                    <p class="text-gray-600">Заполните все поля для отправки уведомления</p>
                </div>
            </div>

            <?php $form = ActiveForm::begin([
                'id' => 'notification-form',
                'options' => ['class' => 'space-y-6']
            ]); ?>

            <!-- Заголовок -->
            <div>
                <?= $form->field($model, 'title', [
                    'template' => '
                        <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                        {input}
                        {error}
                        <div class="text-sm text-gray-500 mt-1">Краткое и понятное название уведомления</div>
                    '
                ])->textInput([
                    'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors text-lg font-medium',
                    'placeholder' => 'Например: Новый конкурс или Изменения в расписании',
                    'id' => 'notification-title'
                ])->label('📌 Заголовок *') ?>
            </div>

            <!-- Сообщение -->
            <div>
                <?= $form->field($model, 'message', [
                    'template' => '
                        <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                        {input}
                        {error}
                        <div class="text-sm text-gray-500 mt-1">Основной текст уведомления</div>
                    '
                ])->textarea([
                    'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                    'rows' => 6,
                    'placeholder' => 'Введите текст уведомления...',
                    'id' => 'notification-message'
                ])->label('📝 Сообщение *') ?>

                <div class="flex justify-between items-center mt-2">
                    <div class="text-xs text-gray-500">
                        <span id="char-count">0</span> / 1000 символов
                    </div>
                    <div class="text-xs text-gray-500">
                        <span id="word-count">0</span> слов
                    </div>
                </div>
            </div>

            <!-- Получатели -->
            <div>
                <h3 class="text-lg font-semibold text-gray-800 mb-4">👥 Кому отправить</h3>
                <div class="space-y-3">
                    <label class="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 cursor-pointer transition-colors">
                        <input type="radio" name="target" value="all" class="h-5 w-5 text-blue-600 border-gray-300 focus:ring-blue-500" checked>
                        <div class="ml-3">
                            <span class="font-medium text-gray-800">Всем пользователям</span>
                            <p class="text-sm text-gray-600">Все зарегистрированные пользователи системы</p>
                            <div class="text-xs text-gray-500 mt-1">
                                👥 <span id="count-all"><?= count($users) ?></span> пользователей
                            </div>
                        </div>
                    </label>

                    <label class="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 cursor-pointer transition-colors">
                        <input type="radio" name="target" value="users" class="h-5 w-5 text-blue-600 border-gray-300 focus:ring-blue-500">
                        <div class="ml-3">
                            <span class="font-medium text-gray-800">Только участникам</span>
                            <p class="text-sm text-gray-600">Пользователи с ролью "участник"</p>
                            <div class="text-xs text-gray-500 mt-1">
                                🎓 <span id="count-users"><?= count(array_filter($users, fn($u) => $u->role === 'participant')) ?></span> участников
                            </div>
                        </div>
                    </label>

                    <label class="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 cursor-pointer transition-colors">
                        <input type="radio" name="target" value="experts" class="h-5 w-5 text-blue-600 border-gray-300 focus:ring-blue-500">
                        <div class="ml-3">
                            <span class="font-medium text-gray-800">Только экспертам</span>
                            <p class="text-sm text-gray-600">Пользователи с ролью "эксперт"</p>
                            <div class="text-xs text-gray-500 mt-1">
                                👨‍⚖️ <span id="count-experts"><?= count(array_filter($users, fn($u) => $u->role === 'expert')) ?></span> экспертов
                            </div>
                        </div>
                    </label>

                    <label class="flex items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 cursor-pointer transition-colors">
                        <input type="radio" name="target" value="specific" class="h-5 w-5 text-blue-600 border-gray-300 focus:ring-blue-500">
                        <div class="ml-3">
                            <span class="font-medium text-gray-800">Конкретному пользователю</span>
                            <p class="text-sm text-gray-600">Выберите одного пользователя из списка</p>
                        </div>
                    </label>
                </div>

                <!-- Выбор конкретного пользователя -->
                <div id="specific-user-select" class="mt-4 hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">👤 Выберите пользователя</label>
                    <select name="Notification[user_id]" class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors">
                        <option value="">-- Выберите пользователя --</option>
                        <?php foreach ($users as $u): ?>
                            <option value="<?= $u->id ?>">
                                <?= Html::encode($u->surname) ?> <?= Html::encode($u->name) ?>
                                (<?= Html::encode($u->role) ?>) - <?= Html::encode($u->email) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <div class="text-sm text-gray-500 mt-1">Будет отправлено только выбранному пользователю</div>
                </div>
            </div>



            <!-- Предварительный просмотр -->
            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <h4 class="text-sm font-medium text-blue-700 mb-2">👁️ Предварительный просмотр</h4>
                <div id="notification-preview" class="border border-gray-300 rounded-lg p-4 bg-white">
                    <div class="flex justify-between items-center mb-3">
                        <h5 id="preview-title" class="font-bold text-gray-800">Заголовок уведомления</h5>
                        <span class="text-xs text-gray-500">только что</span>
                    </div>
                    <p id="preview-message" class="text-sm text-gray-600">
                        Текст уведомления появится здесь...
                    </p>
                    <div class="mt-3 pt-3 border-t border-gray-200 text-xs text-gray-500">
                        Получатели: <span id="preview-recipients" class="font-medium">Все пользователи</span>
                    </div>
                </div>
            </div>

            <!-- Кнопки -->
            <div class="mt-8 flex space-x-4">
                <?= Html::submitButton('🔔 Отправить уведомление', [
                    'class' => 'flex-1 bg-blue-600 text-white py-3 px-6 rounded-lg hover:bg-blue-700 transition-colors font-medium text-lg',
                    'id' => 'submit-btn'
                ]) ?>

                <?= Html::a('Отмена', ['notifications'], [
                    'class' => 'flex-1 border border-gray-300 text-gray-700 py-3 px-6 rounded-lg hover:bg-gray-50 transition-colors text-center font-medium'
                ]) ?>
            </div>

            <?php ActiveForm::end(); ?>
        </div>


    </div>
</div>

<!-- Стили -->
<style>

    /* Стили для выбранного получателя */
    label.flex.items-center input:checked ~ div span.font-medium {
        color: #3b82f6;
    }

    /* Срочное уведомление */
    #notification-preview.urgent {
        border-color: #f87171;
        background-color: #fef2f2;
    }

    /* Адаптивность */
    @media (max-width: 1024px) {
        .lg\:grid-cols-3 {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 768px) {
        .flex.space-x-4 {
            flex-direction: column;
            gap: 0.75rem;
        }

        .flex.space-x-4 > * {
            flex: none;
            width: 100%;
        }
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Элементы DOM
        const titleInput = document.getElementById('notification-title');
        const messageInput = document.getElementById('notification-message');
        const charCount = document.getElementById('char-count');
        const wordCount = document.getElementById('word-count');
        const specificUserSelect = document.getElementById('specific-user-select');
        const userSelect = document.querySelector('select[name="Notification[user_id]"]');
        const urgentCheckbox = document.getElementById('urgent-notification');
        const submitBtn = document.getElementById('submit-btn');

        // Предварительный просмотр
        const previewTitle = document.getElementById('preview-title');
        const previewMessage = document.getElementById('preview-message');
        const previewRecipients = document.getElementById('preview-recipients');
        const previewContainer = document.getElementById('notification-preview');

        // Статистика пользователей
        const countAll = document.getElementById('count-all');
        const countUsers = document.getElementById('count-users');
        const countExperts = document.getElementById('count-experts');

        // Обновление счетчиков символов и слов
        function updateCounters() {
            if (messageInput && charCount) {
                charCount.textContent = messageInput.value.length;

                // Подсчет слов
                const words = messageInput.value.trim().split(/\s+/).filter(word => word.length > 0);
                wordCount.textContent = words.length;

                // Подсветка при превышении лимита
                if (messageInput.value.length > 1000) {
                    charCount.classList.add('text-red-600', 'font-semibold');
                } else {
                    charCount.classList.remove('text-red-600', 'font-semibold');
                }
            }
        }

        // Обновление предварительного просмотра
        function updatePreview() {
            if (titleInput && previewTitle) {
                previewTitle.textContent = titleInput.value || 'Заголовок уведомления';
            }

            if (messageInput && previewMessage) {
                previewMessage.textContent = messageInput.value || 'Текст уведомления появится здесь...';
            }

            // Обновление получателей
            const selectedTarget = document.querySelector('input[name="target"]:checked');
            if (selectedTarget) {
                let recipientsText = '';
                let count = 0;

                switch(selectedTarget.value) {
                    case 'all':
                        recipientsText = `Все пользователи (${countAll.textContent} чел.)`;
                        break;
                    case 'users':
                        recipientsText = `Только участники (${countUsers.textContent} чел.)`;
                        break;
                    case 'experts':
                        recipientsText = `Только эксперты (${countExperts.textContent} чел.)`;
                        break;
                    case 'specific':
                        if (userSelect && userSelect.value) {
                            const selectedOption = userSelect.options[userSelect.selectedIndex];
                            recipientsText = `Конкретный пользователь: ${selectedOption.text}`;
                        } else {
                            recipientsText = 'Конкретный пользователь (не выбран)';
                        }
                        break;
                }

                previewRecipients.textContent = recipientsText;
            }

            // Обновление стиля для срочных уведомлений
            if (urgentCheckbox && previewContainer) {
                if (urgentCheckbox.checked) {
                    previewContainer.classList.add('urgent');
                    previewContainer.querySelector('.font-bold').classList.add('text-red-600');
                } else {
                    previewContainer.classList.remove('urgent');
                    previewContainer.querySelector('.font-bold').classList.remove('text-red-600');
                }
            }
        }

        // Показать/скрыть выбор конкретного пользователя
        function toggleSpecificUserSelect() {
            const selectedTarget = document.querySelector('input[name="target"]:checked');
            if (specificUserSelect) {
                if (selectedTarget && selectedTarget.value === 'specific') {
                    specificUserSelect.classList.remove('hidden');
                    if (userSelect) userSelect.required = true;
                } else {
                    specificUserSelect.classList.add('hidden');
                    if (userSelect) userSelect.required = false;
                }
            }
        }

        // Обработчики событий
        if (titleInput) {
            titleInput.addEventListener('input', updatePreview);
        }

        if (messageInput) {
            messageInput.addEventListener('input', updateCounters);
            messageInput.addEventListener('input', updatePreview);
            // Инициализация счетчиков
            updateCounters();
        }

        if (urgentCheckbox) {
            urgentCheckbox.addEventListener('change', updatePreview);
        }

        // Обработка изменения получателей
        document.querySelectorAll('input[name="target"]').forEach(radio => {
            radio.addEventListener('change', function() {
                toggleSpecificUserSelect();
                updatePreview();

                // Обновление стилей выбранного варианта
                document.querySelectorAll('label.flex.items-center').forEach(label => {
                    label.classList.remove('border-blue-300', 'bg-blue-50');
                });
                if (this.checked) {
                    this.closest('label').classList.add('border-blue-300', 'bg-blue-50');
                }
            });
        });

        if (userSelect) {
            userSelect.addEventListener('change', updatePreview);
        }

        // Шаблоны уведомлений
        const templateButtons = document.querySelectorAll('button[data-template]');
        templateButtons.forEach(button => {
            button.addEventListener('click', function() {
                const template = this.getAttribute('data-template');
                const templates = {
                    'contest-start': {
                        title: '🏆 Начался новый конкурс!',
                        message: 'Уважаемые участники!\n\nМы рады сообщить о начале нового конкурса. Регистрация открыта до [дата].\n\nВсе подробности и условия участия вы найдете на странице конкурса.\n\nЖелаем творческих успехов!'
                    },
                    'results': {
                        title: '📊 Результаты оценки готовы',
                        message: 'Уважаемые участники!\n\nОценка конкурсных работ завершена. Результаты доступны в вашем личном кабинете.\n\nПоздравляем победителей и благодарим всех за участие!'
                    },
                    'system': {
                        title: '⚙️ Технические работы',
                        message: 'Уважаемые пользователи!\n\nВ связи с проведением технических работ система будет недоступна [дата] с [время] до [время].\n\nПриносим извинения за возможные неудобства.'
                    },
                    'reminder': {
                        title: '⏰ Напоминание: срок подачи заявок',
                        message: 'Уважаемые участники!\n\nНапоминаем, что срок подачи заявок на конкурс "[Название конкурса]" истекает [дата].\n\nУспейте подать свою работу вовремя!'
                    }
                };

                const templateData = templates[template];
                if (templateData) {
                    if (titleInput) titleInput.value = templateData.title;
                    if (messageInput) messageInput.value = templateData.message;

                    updateCounters();
                    updatePreview();

                    // Прокрутка к форме
                    document.querySelector('form').scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // Валидация формы
        const form = document.getElementById('notification-form');
        if (form) {
            form.addEventListener('submit', function(e) {
                // Проверка обязательных полей
                if (!titleInput || !titleInput.value.trim()) {
                    e.preventDefault();
                    alert('Пожалуйста, введите заголовок уведомления');
                    titleInput.focus();
                    titleInput.classList.add('border-red-500');
                    return false;
                }

                if (!messageInput || !messageInput.value.trim()) {
                    e.preventDefault();
                    alert('Пожалуйста, введите текст уведомления');
                    messageInput.focus();
                    messageInput.classList.add('border-red-500');
                    return false;
                }

                // Проверка длины сообщения
                if (messageInput.value.length > 1000) {
                    if (!confirm('Сообщение превышает 1000 символов. Уведомление может быть обрезано. Всё равно отправить?')) {
                        e.preventDefault();
                        messageInput.focus();
                        return false;
                    }
                }

                // Проверка выбора конкретного пользователя
                const selectedTarget = document.querySelector('input[name="target"]:checked');
                if (selectedTarget && selectedTarget.value === 'specific' && userSelect && !userSelect.value) {
                    e.preventDefault();
                    alert('Пожалуйста, выберите конкретного пользователя');
                    userSelect.focus();
                    userSelect.classList.add('border-red-500');
                    return false;
                }

                // Подтверждение массовой рассылки
                const targetValue = selectedTarget ? selectedTarget.value : 'all';
                if (targetValue === 'all' || targetValue === 'users' || targetValue === 'experts') {
                    const count = targetValue === 'all' ? countAll.textContent :
                        targetValue === 'users' ? countUsers.textContent : countExperts.textContent;

                    if (!confirm(`Вы собираетесь отправить уведомление ${count} пользователям. Продолжить?`)) {
                        e.preventDefault();
                        return false;
                    }
                }

                // Показать индикатор загрузки
                if (submitBtn) {
                    const originalText = submitBtn.innerHTML;
                    submitBtn.innerHTML = '<span class="flex items-center justify-center"><svg class="animate-spin h-5 w-5 mr-2 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg> Отправка...</span>';
                    submitBtn.disabled = true;

                    setTimeout(() => {
                        submitBtn.innerHTML = originalText;
                        submitBtn.disabled = false;
                    }, 3000);
                }
            });
        }

        // Сброс ошибок при изменении
        if (titleInput) {
            titleInput.addEventListener('input', function() {
                this.classList.remove('border-red-500');
            });
        }

        if (messageInput) {
            messageInput.addEventListener('input', function() {
                this.classList.remove('border-red-500');
            });
        }

        if (userSelect) {
            userSelect.addEventListener('change', function() {
                this.classList.remove('border-red-500');
            });
        }

        // Инициализация
        toggleSpecificUserSelect();
        updatePreview();

        // Выделяем первый выбранный вариант
        const firstRadio = document.querySelector('input[name="target"]:checked');
        if (firstRadio) {
            firstRadio.closest('label').classList.add('border-blue-300', 'bg-blue-50');
        }
    });
</script>