<?php
use yii\helpers\Html;
use yii\widgets\ActiveForm;
?>

<!-- Основной контент -->
<div id="nomination-update" class="page">
    <div class="flex justify-between items-center mb-8">
        <div>
            <h2 class="text-3xl font-bold text-gray-800">Редактирование номинации</h2>
            <p class="text-gray-600 mt-2">
                ID: <span class="font-semibold">#<?= $model->id ?></span> |
                Текущее название: <span class="font-semibold"><?= Html::encode($model->name) ?></span>
            </p>
        </div>
        <?= Html::a('← Назад к номинации', ['nomination-view', 'id' => $model->id], [
            'class' => 'text-blue-600 hover:text-blue-800 flex items-center'
        ]) ?>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- Основная форма -->
        <div class="bg-white rounded-xl shadow-md p-6 lg:col-span-2">
            <div class="flex items-center mb-6">
                <div class="p-3 bg-blue-100 rounded-lg mr-4">
                    <span class="text-2xl text-blue-600">
                        <?php
                        $icons = ['🎭', '🎨', '🎤', '💃', '🎼', '🎬', '📸', '✍️'];
                        echo $model->icon ?? $icons[$model->id % count($icons)] ?? '🏆';
                        ?>
                    </span>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800">✏️ Редактирование данных</h3>
                    <p class="text-gray-600">Обновите информацию о номинации</p>
                </div>
            </div>

            <?php $form = ActiveForm::begin([
                'id' => 'nomination-form',
                'options' => ['class' => 'space-y-6']
            ]); ?>

            <!-- Название номинации -->
            <div>
                <?= $form->field($model, 'name', [
                    'template' => '
                        <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                        {input}
                        {error}
                       
                    '
                ])->textInput([
                    'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors text-lg font-medium',
                    'placeholder' => 'Введите новое название номинации',
                    'id' => 'nomination-name'
                ])->label('🎯 Название номинации *') ?>
            </div>

            <!-- Описание (если есть в модели) -->
            <?php if (isset($model->description)): ?>
                <div>
                    <?= $form->field($model, 'description', [
                        'template' => '
                        <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                        {input}
                        {error}
                        <div class="text-sm text-gray-500 mt-1">Измените описание номинации</div>
                    '
                    ])->textarea([
                        'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                        'rows' => 4,
                        'placeholder' => 'Опишите критерии и особенности номинации...',
                        'id' => 'nomination-description'
                    ])->label('📝 Описание номинации') ?>

                    <div class="text-xs text-gray-500 text-right mt-1">
                        <span id="char-count"><?= strlen($model->description ?? '') ?></span> / 500 символов
                    </div>
                </div>
            <?php endif; ?>

            <!-- Иконка (если есть в модели) -->
            <?php if (isset($model->icon)): ?>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">🎨 Иконка номинации</label>
                    <p class="text-sm text-gray-500 mb-4">Текущая иконка: <span class="text-xl"><?= $model->icon ?? '🏆' ?></span></p>
                    <div class="grid grid-cols-4 md:grid-cols-8 gap-2 mb-4">
                        <?php
                        $icons = ['🎭', '🎨', '🎤', '💃', '🎼', '🎬', '📸', '✍️', '🎹', '🎻', '🎷', '🥁', '🩰', '🎪', '🌟', '🏅'];
                        $currentIcon = $model->icon ?? '';
                        foreach ($icons as $icon): ?>
                            <button type="button" class="p-3 border <?= $icon === $currentIcon ? 'border-blue-500 bg-blue-50' : 'border-gray-300' ?> rounded-lg hover:bg-gray-50 text-xl transition-colors icon-select <?= $icon === $currentIcon ? 'selected' : '' ?>" data-icon="<?= $icon ?>">
                                <?= $icon ?>
                            </button>
                        <?php endforeach; ?>
                    </div>
                    <?= $form->field($model, 'icon', [
                        'template' => '{input}{error}'
                    ])->hiddenInput(['id' => 'nomination-icon']) ?>
                </div>
            <?php endif; ?>

            <!-- Критерии оценки (если есть) -->
            <?php if (isset($model->criteria)): ?>
                <div>
                    <?= $form->field($model, 'criteria', [
                        'template' => '
                        <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                        {input}
                        {error}
                        <div class="text-sm text-gray-500 mt-1">Критерии оценки через запятую</div>
                    '
                    ])->textInput([
                        'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                        'placeholder' => 'Техника, Артистизм, Оригинальность, Сложность'
                    ])->label('📊 Критерии оценки') ?>
                </div>
            <?php endif; ?>

            <!-- Возрастные ограничения (если есть) -->
            <?php if (isset($model->min_age) || isset($model->max_age)): ?>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <?php if (isset($model->min_age)): ?>
                        <div>
                            <?= $form->field($model, 'min_age', [
                                'template' => '
                            <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                            {input}
                            {error}
                        '
                            ])->textInput([
                                'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                                'type' => 'number',
                                'placeholder' => '6'
                            ])->label('👶 Минимальный возраст') ?>
                        </div>
                    <?php endif; ?>

                    <?php if (isset($model->max_age)): ?>
                        <div>
                            <?= $form->field($model, 'max_age', [
                                'template' => '
                            <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                            {input}
                            {error}
                        '
                            ])->textInput([
                                'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                                'type' => 'number',
                                'placeholder' => '18'
                            ])->label('👴 Максимальный возраст') ?>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>



            <!-- Кнопки -->
            <div class="mt-8 flex space-x-4">
                <?= Html::submitButton('💾 Сохранить изменения', [
                    'class' => 'flex-1 bg-blue-600 text-white py-3 px-6 rounded-lg hover:bg-blue-700 transition-colors font-medium text-lg',
                    'id' => 'submit-btn'
                ]) ?>
            </div>

            <?php ActiveForm::end(); ?>
        </div>


    </div>
</div>

<!-- Стили -->
<style>

    /* Выбранная иконка */
    .icon-select.selected {
        background-color: #3b82f6;
        color: white;
        border-color: #3b82f6;
    }

    /* Ошибки валидации */
    .has-error .form-control {
        border-color: #f87171;
        background-color: #fef2f2;
    }

    .has-error .help-block {
        color: #dc2626;
        font-size: 0.875rem;
        margin-top: 0.25rem;
    }

    /* Адаптивность */
    @media (max-width: 1024px) {
        .lg\:grid-cols-3 {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 768px) {
        .grid-cols-2, .md\:grid-cols-8 {
            grid-template-columns: repeat(4, 1fr);
        }

        .flex.space-x-4 {
            flex-direction: column;
            gap: 0.75rem;
        }

        .flex.space-x-4 > * {
            flex: none;
            width: 100%;
        }
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Элементы DOM
        const nameInput = document.getElementById('nomination-name');
        const descriptionInput = document.getElementById('nomination-description');
        const iconInput = document.getElementById('nomination-icon');
        const charCount = document.getElementById('char-count');
        const submitBtn = document.getElementById('submit-btn');
        const originalName = '<?= addslashes($model->name) ?>';

        // Предварительный просмотр
        const previewName = document.getElementById('preview-name');
        const previewDescription = document.getElementById('preview-description');
        const previewIcon = document.getElementById('preview-icon');

        // Сохраняем оригинальные значения
        const originalValues = {
            name: nameInput ? nameInput.value : '',
            description: descriptionInput ? descriptionInput.value : '',
            icon: iconInput ? iconInput.value : ''
        };

        // Обновление предварительного просмотра
        function updatePreview() {
            if (nameInput && previewName) {
                previewName.textContent = nameInput.value || originalValues.name;
            }

            if (descriptionInput && previewDescription) {
                previewDescription.textContent = descriptionInput.value || originalValues.description || 'Описание отсутствует';
                if (descriptionInput.value.length > 100) {
                    previewDescription.textContent = descriptionInput.value.substring(0, 100) + '...';
                }
            }

            if (charCount && descriptionInput) {
                charCount.textContent = descriptionInput.value.length;
            }
        }

        // Проверка изменений
        function hasChanges() {
            if (nameInput && nameInput.value !== originalValues.name) return true;
            if (descriptionInput && descriptionInput.value !== originalValues.description) return true;
            if (iconInput && iconInput.value !== originalValues.icon) return true;
            return false;
        }

        // Обработчики событий
        if (nameInput) {
            nameInput.addEventListener('input', updatePreview);
            nameInput.addEventListener('input', function() {
                if (this.value !== originalValues.name) {
                    this.classList.add('border-blue-400', 'bg-blue-50');
                } else {
                    this.classList.remove('border-blue-400', 'bg-blue-50');
                }
            });
        }

        if (descriptionInput) {
            descriptionInput.addEventListener('input', updatePreview);
            descriptionInput.addEventListener('input', function() {
                if (this.value !== originalValues.description) {
                    this.classList.add('border-blue-400', 'bg-blue-50');
                } else {
                    this.classList.remove('border-blue-400', 'bg-blue-50');
                }
                if (charCount) {
                    charCount.textContent = this.value.length;
                }
            });
            // Инициализация счетчика символов
            if (charCount) {
                charCount.textContent = descriptionInput.value.length;
            }
        }

        // Выбор иконки
        const iconButtons = document.querySelectorAll('.icon-select');
        iconButtons.forEach(button => {
            button.addEventListener('click', function() {
                // Сброс предыдущего выбора
                iconButtons.forEach(btn => btn.classList.remove('selected'));

                // Установка нового выбора
                this.classList.add('selected');
                const icon = this.getAttribute('data-icon');

                if (iconInput) {
                    iconInput.value = icon;
                    if (icon !== originalValues.icon) {
                        iconInput.classList.add('changed');
                    } else {
                        iconInput.classList.remove('changed');
                    }
                }

                if (previewIcon) {
                    previewIcon.textContent = icon;
                }
            });
        });

        // Валидация формы
        const form = document.getElementById('nomination-form');
        if (form) {
            form.addEventListener('submit', function(e) {
                if (!nameInput || !nameInput.value.trim()) {
                    e.preventDefault();
                    alert('Пожалуйста, укажите название номинации');
                    nameInput.focus();
                    nameInput.classList.add('border-red-500');
                    return false;
                }

                // Проверка на наличие изменений
                if (!hasChanges()) {
                    if (!confirm('Вы не внесли никаких изменений. Всё равно сохранить?')) {
                        e.preventDefault();
                        return false;
                    }
                }

                // Проверка на изменение названия
                if (nameInput && nameInput.value !== originalName) {
                    if (!confirm(`Вы собираетесь изменить название номинации с "${originalName}" на "${nameInput.value}". Это повлияет на все связанные документы. Продолжить?`)) {
                        e.preventDefault();
                        return false;
                    }
                }

                // Показать индикатор загрузки
                if (submitBtn) {
                    const originalText = submitBtn.innerHTML;
                    submitBtn.innerHTML = '<span class="flex items-center justify-center"><svg class="animate-spin h-5 w-5 mr-2 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg> Сохранение...</span>';
                    submitBtn.disabled = true;

                    setTimeout(() => {
                        submitBtn.innerHTML = originalText;
                        submitBtn.disabled = false;
                    }, 3000);
                }
            });
        }

        // Защита от случайного закрытия
        window.addEventListener('beforeunload', function(e) {
            if (hasChanges()) {
                e.preventDefault();
                e.returnValue = 'У вас есть несохранённые изменения. Вы уверены, что хотите покинуть страницу?';
                return 'У вас есть несохранённые изменения. Вы уверены, что хотите покинуть страницу?';
            }
        });

        // Инициализация предварительного просмотра
        updatePreview();

        // Подтверждение удаления
        const deleteLink = document.querySelector('a[data-confirm]');
        if (deleteLink) {
            deleteLink.addEventListener('click', function(e) {
                if (!confirm(this.getAttribute('data-confirm'))) {
                    e.preventDefault();
                    return false;
                }


                }
            });
        }
    });
</script>