<?php
use yii\helpers\Html;
?>

<!-- Основной контент -->
<div id="age-categories-index" class="page">
    <div class="flex justify-between items-center mb-8">
        <h2 class="text-3xl font-bold text-gray-800">👶 Управление возрастными категориями</h2>
        <?= Html::a('➕ Создать категорию', ['age-category-create'], [
            'class' => 'bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors flex items-center'
        ]) ?>
    </div>



    <!-- Таблица категорий -->
    <div class="bg-white rounded-xl shadow-md p-6">
        <div class="flex justify-between items-center mb-6">
            <h3 class="text-2xl font-bold text-gray-800">Список возрастных категорий</h3>
        </div>

        <div class="overflow-x-auto">
            <table class="w-full">
                <thead>
                <tr class="border-b border-gray-200">
                    <th class="text-left py-3 font-semibold text-gray-700">#</th>
                    <th class="text-left py-3 font-semibold text-gray-700">Категория</th>
                    <th class="text-left py-3 font-semibold text-gray-700">Возрастной диапазон</th>
                    <th class="text-left py-3 font-semibold text-gray-700">Статистика</th>
                    <th class="text-left py-3 font-semibold text-gray-700">Описание</th>
                    <th class="text-left py-3 font-semibold text-gray-700">Действия</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach ($dataProvider->getModels() as $index => $category): ?>
                    <?php
                    // Парсим возраст из названия
                    $minAge = null;
                    $maxAge = null;
                    $ageGroup = 'other';

                    if (preg_match('/(\d+)-(\d+)/', $category->name, $matches)) {
                        $minAge = (int)$matches[1];
                        $maxAge = (int)$matches[2];
                        if ($maxAge <= 12) $ageGroup = 'children';
                        elseif ($maxAge <= 18) $ageGroup = 'youth';
                        else $ageGroup = 'adults';
                    }

                    // Определяем иконку и цвет
                    $icons = [
                        'children' => ['👶', 'bg-blue-100 text-blue-800'],
                        'youth' => ['👦', 'bg-green-100 text-green-800'],
                        'adults' => ['👨', 'bg-purple-100 text-purple-800'],
                        'other' => ['👤', 'bg-gray-100 text-gray-800']
                    ];
                    [$icon, $colorClass] = $icons[$ageGroup];
                    ?>
                    <tr class="border-b border-gray-100 hover:bg-gray-50 transition-colors" data-age-group="<?= $ageGroup ?>">
                        <td class="py-4 text-gray-600"><?= $index + 1 + ($dataProvider->pagination->page * $dataProvider->pagination->pageSize) ?></td>

                        <!-- Название категории -->
                        <td class="py-4">
                            <div class="flex items-center">
                                <div class="p-2 <?= $colorClass ?> rounded-lg mr-3">
                                    <span class="text-xl"><?= $icon ?></span>
                                </div>
                                <div>
                                    <h4 class="font-bold text-gray-800 text-lg"><?= Html::encode($category->name) ?></h4>
                                    <div class="text-sm text-gray-500">
                                        ID: #<?= $category->id ?>
                                    </div>
                                </div>
                            </div>
                        </td>

                        <!-- Возрастной диапазон -->
                        <td class="py-4">
                            <?php if ($minAge !== null && $maxAge !== null): ?>
                                <div class="space-y-2">
                                    <div class="flex items-center">
                                        <span class="text-gray-600 mr-2">От:</span>
                                        <span class="font-bold text-blue-600"><?= $minAge ?> лет</span>
                                    </div>
                                    <div class="flex items-center">
                                        <span class="text-gray-600 mr-2">До:</span>
                                        <span class="font-bold text-blue-600"><?= $maxAge ?> лет</span>
                                    </div>
                                    <div class="w-full bg-gray-200 rounded-full h-2">
                                        <div class="bg-blue-600 h-2 rounded-full" style="width: <?= min(100, ($minAge / 30) * 100) ?>%"></div>
                                    </div>
                                </div>
                            <?php else: ?>
                                <span class="text-gray-400 italic">Не указан</span>
                            <?php endif; ?>
                        </td>

                        <!-- Статистика -->
                        <td class="py-4">
                            <div class="space-y-2">
                                <?php
                                $applicationsCount = $category->getApplications()->count();
                                $activeApplications = $category->getApplications()->andWhere(['status' => ['new', 'accepted']])->count();
                                ?>
                                <div class="flex items-center text-sm">
                                    <span class="text-gray-600 mr-2">Заявок:</span>
                                    <span class="font-bold <?= $applicationsCount > 0 ? 'text-green-600' : 'text-gray-600' ?>">
                                        <?= $applicationsCount ?>
                                    </span>
                                </div>
                                <div class="flex items-center text-sm">
                                    <span class="text-gray-600 mr-2">Активных:</span>
                                    <span class="font-bold <?= $activeApplications > 0 ? 'text-blue-600' : 'text-gray-600' ?>">
                                        <?= $activeApplications ?>
                                    </span>
                                </div>
                            </div>
                        </td>

                        <!-- Описание -->
                        <td class="py-4">
                            <div class="text-gray-600 max-w-xs">
                                <?php if (!empty($category->description)): ?>
                                    <p class="text-sm"><?= Html::encode(substr($category->description, 0, 80)) ?><?= strlen($category->description) > 80 ? '...' : '' ?></p>
                                <?php else: ?>
                                    <p class="text-sm text-gray-400 italic">Описание отсутствует</p>
                                <?php endif; ?>
                            </div>
                        </td>

                        <!-- Действия -->
                        <td class="py-4">
                            <div class="flex space-x-3">

                                <?= Html::a('✏️', ['age-category-update', 'id' => $category->id], [
                                    'class' => 'text-green-600 hover:text-green-800 text-lg',
                                    'title' => 'Редактировать'
                                ]) ?>

                                <?= Html::a('🗑️', ['age-category-delete', 'id' => $category->id], [
                                    'class' => 'text-red-600 hover:text-red-800 text-lg',
                                    'title' => 'Удалить',
                                    'data' => [
                                        'confirm' => 'Вы уверены, что хотите удалить категорию "' . Html::encode($category->name) . '"? Это может затронуть существующие заявки.'
                                    ]
                                ]) ?>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Пагинация -->
        <?php if ($dataProvider->pagination->pageCount > 1): ?>
            <div class="mt-6 flex justify-between items-center">
                <div class="text-sm text-gray-600">
                    Показано <?= $dataProvider->pagination->pageSize * $dataProvider->pagination->page + 1 ?>-
                    <?= min($dataProvider->pagination->pageSize * ($dataProvider->pagination->page + 1), $dataProvider->totalCount) ?>
                    из <?= $dataProvider->totalCount ?> категорий
                </div>
                <div class="flex space-x-2">
                    <?php if ($dataProvider->pagination->page > 0): ?>
                        <?= Html::a('« Назад', ['index', 'page' => $dataProvider->pagination->page], [
                            'class' => 'px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50'
                        ]) ?>
                    <?php endif; ?>

                    <?php for ($i = 0; $i < $dataProvider->pagination->pageCount; $i++): ?>
                        <?php if (abs($i - $dataProvider->pagination->page) <= 2): ?>
                            <?= Html::a($i + 1, ['index', 'page' => $i], [
                                'class' => 'px-4 py-2 rounded-lg ' . ($i == $dataProvider->pagination->page
                                        ? 'bg-blue-600 text-white'
                                        : 'border border-gray-300 hover:bg-gray-50')
                            ]) ?>
                        <?php endif; ?>
                    <?php endfor; ?>

                    <?php if ($dataProvider->pagination->page < $dataProvider->pagination->pageCount - 1): ?>
                        <?= Html::a('Вперед »', ['index', 'page' => $dataProvider->pagination->page + 2], [
                            'class' => 'px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50'
                        ]) ?>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <!-- Пустое состояние -->
    <?php if ($dataProvider->totalCount == 0): ?>
        <div class="bg-yellow-50 border border-yellow-200 rounded-xl p-8 text-center mt-6">
            <div class="text-6xl mb-4">👶</div>
            <h3 class="text-2xl font-bold text-yellow-800 mb-2">Возрастных категорий пока нет</h3>
            <p class="text-yellow-600 mb-6">Создайте первую возрастную категорию для организации участников по возрастам</p>
            <?= Html::a('➕ Создать первую категорию', ['age-category-create'], [
                'class' => 'bg-yellow-500 text-white px-6 py-3 rounded-lg hover:bg-yellow-600 transition-colors inline-flex items-center'
            ]) ?>
        </div>
    <?php endif; ?>


</div>

<!-- Стили -->
<style>

    /* Адаптивность */
    @media (max-width: 768px) {
        table {
            min-width: 900px;
        }

        .flex.justify-between {
            flex-direction: column;
            gap: 1rem;
        }

        .grid-cols-3 {
            grid-template-columns: 1fr;
        }

        .flex.space-x-2, .flex.space-x-3 {
            flex-wrap: wrap;
            gap: 0.5rem;
        }

        .md\:grid-cols-3 {
            grid-template-columns: 1fr;
        }
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Фильтрация по возрастным группам
        const filterAll = document.getElementById('filter-all');
        const filterChildren = document.getElementById('filter-children');
        const filterYouth = document.getElementById('filter-youth');
        const filterAdults = document.getElementById('filter-adults');
        const tableRows = document.querySelectorAll('tbody tr');

        function resetFilters() {
            [filterAll, filterChildren, filterYouth, filterAdults].forEach(btn => {
                btn.classList.remove('bg-blue-600', 'text-white');
                btn.classList.add('border', 'border-gray-300', 'text-gray-700');
            });
        }

        filterAll.addEventListener('click', function() {
            resetFilters();
            this.classList.remove('border', 'border-gray-300', 'text-gray-700');
            this.classList.add('bg-blue-600', 'text-white');

            tableRows.forEach(row => row.style.display = '');
        });

        filterChildren.addEventListener('click', function() {
            resetFilters();
            this.classList.remove('border', 'border-gray-300', 'text-gray-700');
            this.classList.add('bg-blue-600', 'text-white');

            tableRows.forEach(row => {
                const ageGroup = row.getAttribute('data-age-group');
                row.style.display = ageGroup === 'children' ? '' : 'none';
            });
        });

        filterYouth.addEventListener('click', function() {
            resetFilters();
            this.classList.remove('border', 'border-gray-300', 'text-gray-700');
            this.classList.add('bg-blue-600', 'text-white');

            tableRows.forEach(row => {
                const ageGroup = row.getAttribute('data-age-group');
                row.style.display = ageGroup === 'youth' ? '' : 'none';
            });
        });

        filterAdults.addEventListener('click', function() {
            resetFilters();
            this.classList.remove('border', 'border-gray-300', 'text-gray-700');
            this.classList.add('bg-blue-600', 'text-white');

            tableRows.forEach(row => {
                const ageGroup = row.getAttribute('data-age-group');
                row.style.display = ageGroup === 'adults' ? '' : 'none';
            });
        });

        // Подтверждение удаления
        const deleteLinks = document.querySelectorAll('a[data-confirm]');
        deleteLinks.forEach(link => {
            link.addEventListener('click', function(e) {
                if (!confirm(this.getAttribute('data-confirm'))) {
                    e.preventDefault();
                    return false;
                }

                // Дополнительная проверка если есть заявки
                const row = this.closest('tr');
                const applicationsCount = parseInt(row.querySelector('.text-green-600')?.textContent || 0);
                if (applicationsCount > 0) {
                    if (!confirm(`Внимание! У этой категории есть ${applicationsCount} связанных заявок. Они останутся без возрастной категории. Продолжить удаление?`)) {
                        e.preventDefault();
                        return false;
                    }
                }

                // Показать индикатор загрузки
                const originalHTML = this.innerHTML;
                this.innerHTML = '<span class="flex items-center"><svg class="animate-spin h-4 w-4 mr-1" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg> Удаление...</span>';
                this.classList.add('opacity-50');

                setTimeout(() => {
                    this.innerHTML = originalHTML;
                    this.classList.remove('opacity-50');
                }, 2000);
            });
        });

        // Подсветка популярных категорий
        tableRows.forEach(row => {
            const applicationsCount = parseInt(row.querySelector('.text-green-600')?.textContent || 0);
            if (applicationsCount >= 10) {
                row.classList.add('bg-gradient-to-r', 'from-yellow-50', 'to-transparent');
                const popularBadge = document.createElement('span');
                popularBadge.className = 'ml-2 text-xs bg-yellow-500 text-white px-2 py-1 rounded-full';
                popularBadge.textContent = '🔥 Популярная';
                row.querySelector('.font-bold.text-gray-800').appendChild(popularBadge);
            }
        });

        // Автоматическое определение возрастной группы при загрузке
        let hasChildren = false, hasYouth = false, hasAdults = false;
        tableRows.forEach(row => {
            const ageGroup = row.getAttribute('data-age-group');
            if (ageGroup === 'children') hasChildren = true;
            if (ageGroup === 'youth') hasYouth = true;
            if (ageGroup === 'adults') hasAdults = true;
        });

        // Деактивация кнопок фильтров если нет соответствующих категорий
        if (!hasChildren) filterChildren.disabled = true;
        if (!hasYouth) filterYouth.disabled = true;
        if (!hasAdults) filterAdults.disabled = true;
    });
</script>