<?php
use yii\helpers\Html;
use yii\widgets\ActiveForm;
?>

<!-- Основной контент -->
<div id="age-category-create" class="page">
    <div class="flex justify-between items-center mb-8">
        <h2 class="text-3xl font-bold text-gray-800">Создание возрастной категории</h2>
        <?= Html::a('← Назад к списку', ['index'], [
            'class' => 'text-blue-600 hover:text-blue-800 flex items-center'
        ]) ?>
    </div>

    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- Основная форма -->
        <div class="bg-white rounded-xl shadow-md p-6 lg:col-span-2">
            <div class="flex items-center mb-6">
                <div class="p-3 bg-blue-100 rounded-lg mr-4">
                    <span class="text-2xl text-blue-600">👶</span>
                </div>
                <div>
                    <h3 class="text-2xl font-bold text-gray-800">Добавить новую возрастную категорию</h3>
                    <p class="text-gray-600">Создайте категорию для группировки участников по возрасту</p>
                </div>
            </div>

            <?php $form = ActiveForm::begin([
                'id' => 'age-category-form',
                'options' => ['class' => 'space-y-6']
            ]); ?>

            <!-- Название категории -->
            <div>
                <?= $form->field($model, 'name', [
                    'template' => '
                        <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                        {input}
                        {error}
                    '
                ])->textInput([
                    'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors text-lg font-medium',
                    'placeholder' => 'Например: Дети 6-9 лет',
                    'id' => 'category-name'
                ])->label('🎯 Название категории *') ?>

                <!-- Автоматическое определение возраста -->
                <div id="age-detection" class="hidden mt-2 p-3 bg-blue-50 border border-blue-200 rounded-lg">
                    <div class="flex items-center text-sm text-blue-700">
                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                        <span>Определен возраст: от <span id="min-age">0</span> до <span id="max-age">0</span> лет</span>
                    </div>
                </div>
            </div>

            <!-- Описание (если есть в модели) -->
            <?php if (isset($model->description)): ?>
                <div>
                    <?= $form->field($model, 'description', [
                        'template' => '
                        <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                        {input}
                        {error}
                        <div class="text-sm text-gray-500 mt-1">Опишите особенности этой возрастной группы</div>
                    '
                    ])->textarea([
                        'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                        'rows' => 3,
                        'placeholder' => 'Например: Младшая школьная группа, начальный уровень подготовки...',
                        'id' => 'category-description'
                    ])->label('📝 Описание категории') ?>
                </div>
            <?php endif; ?>

            <!-- Минимальный и максимальный возраст (если есть отдельные поля) -->
            <?php if (isset($model->min_age) && isset($model->max_age)): ?>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <?= $form->field($model, 'min_age', [
                            'template' => '
                            <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                            {input}
                            {error}
                        '
                        ])->textInput([
                            'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                            'type' => 'number',
                            'placeholder' => '6',
                            'min' => '0',
                            'max' => '100',
                            'id' => 'min-age-input'
                        ])->label('👶 Минимальный возраст') ?>
                    </div>

                    <div>
                        <?= $form->field($model, 'max_age', [
                            'template' => '
                            <label class="block text-sm font-medium text-gray-700 mb-2">{label}</label>
                            {input}
                            {error}
                        '
                        ])->textInput([
                            'class' => 'w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors',
                            'type' => 'number',
                            'placeholder' => '9',
                            'min' => '0',
                            'max' => '100',
                            'id' => 'max-age-input'
                        ])->label('👴 Максимальный возраст') ?>
                    </div>
                </div>

                <!-- Визуализация возраста -->
                <div class="bg-gray-50 border border-gray-200 rounded-lg p-4">
                    <div class="flex justify-between text-sm text-gray-600 mb-2">
                        <span>0 лет</span>
                        <span>Диапазон возраста</span>
                        <span>18+ лет</span>
                    </div>
                    <div class="w-full bg-gray-200 rounded-full h-3">
                        <div id="age-range-visual" class="bg-blue-600 h-3 rounded-full" style="width: 0%"></div>
                    </div>
                    <div class="mt-2 text-center">
                        <span id="age-range-text" class="text-sm font-medium text-gray-700">Выберите возраст</span>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Иконка (если есть в модели) -->
            <?php if (isset($model->icon)): ?>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">🎨 Иконка категории</label>
                    <p class="text-sm text-gray-500 mb-4">Выберите подходящую иконку для возрастной группы</p>
                    <div class="grid grid-cols-4 md:grid-cols-6 gap-2 mb-4">
                        <?php
                        $icons = [
                            '👶' => 'Младенцы/Дети',
                            '👧' => 'Девочки',
                            '👦' => 'Мальчики',
                            '🧒' => 'Дети',
                            '👩' => 'Женщины',
                            '👨' => 'Мужчины',
                            '🧑' => 'Взрослые',
                            '👵' => 'Пожилые',
                            '👴' => 'Пожилые',
                            '👱' => 'Молодые',
                            '🧔' => 'Взрослые',
                            '👨‍🎓' => 'Студенты'
                        ];
                        foreach ($icons as $icon => $title): ?>
                            <button type="button" class="p-3 border border-gray-300 rounded-lg hover:bg-gray-50 text-xl transition-colors icon-select flex flex-col items-center" data-icon="<?= $icon ?>" title="<?= $title ?>">
                                <?= $icon ?>
                                <span class="text-xs text-gray-500 mt-1"><?= $title ?></span>
                            </button>
                        <?php endforeach; ?>
                    </div>
                    <?= $form->field($model, 'icon', [
                        'template' => '{input}{error}'
                    ])->hiddenInput(['id' => 'category-icon']) ?>
                </div>
            <?php endif; ?>

            <!-- Кнопки -->
            <div class="mt-8 flex space-x-4">
                <?= Html::submitButton('👶 Создать категорию', [
                    'class' => 'flex-1 bg-blue-600 text-white py-3 px-6 rounded-lg hover:bg-blue-700 transition-colors font-medium text-lg',
                    'id' => 'submit-btn'
                ]) ?>

                <?= Html::a('Отмена', ['index'], [
                    'class' => 'flex-1 border border-gray-300 text-gray-700 py-3 px-6 rounded-lg hover:bg-gray-50 transition-colors text-center font-medium'
                ]) ?>
            </div>

            <?php ActiveForm::end(); ?>
        </div>


    </div>
</div>

<!-- Стили -->
<style>
    /* Выбранная иконка */
    .icon-select.selected {
        background-color: #3b82f6;
        color: white;
        border-color: #3b82f6;
    }

    .icon-select.selected span {
        color: #dbeafe;
    }

    /* Ошибки валидации */
    .has-error .form-control {
        border-color: #f87171;
        background-color: #fef2f2;
    }

    .has-error .help-block {
        color: #dc2626;
        font-size: 0.875rem;
        margin-top: 0.25rem;
    }

    /* Адаптивность */
    @media (max-width: 1024px) {
        .lg\:grid-cols-3 {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 768px) {
        .grid-cols-2, .md\:grid-cols-6 {
            grid-template-columns: repeat(3, 1fr);
        }

        .flex.space-x-4 {
            flex-direction: column;
            gap: 0.75rem;
        }

        .flex.space-x-4 > * {
            flex: none;
            width: 100%;
        }
    }
</style>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Элементы DOM
        const nameInput = document.getElementById('category-name');
        const descriptionInput = document.getElementById('category-description');
        const iconInput = document.getElementById('category-icon');
        const minAgeInput = document.getElementById('min-age-input');
        const maxAgeInput = document.getElementById('max-age-input');
        const submitBtn = document.getElementById('submit-btn');
        const ageDetection = document.getElementById('age-detection');

        // Предварительный просмотр
        const previewName = document.getElementById('preview-name');
        const previewDescription = document.getElementById('preview-description');
        const previewIcon = document.getElementById('preview-icon');
        const previewAge = document.getElementById('preview-age');
        const ageRangeVisual = document.getElementById('age-range-visual');
        const ageRangeText = document.getElementById('age-range-text');

        // Обновление предварительного просмотра
        function updatePreview() {
            if (nameInput && previewName) {
                previewName.textContent = nameInput.value || 'Название категории';

                // Автоматическое определение возраста из названия
                const ageMatch = nameInput.value.match(/(\d+)[-\s](\d+)/);
                if (ageMatch) {
                    const min = parseInt(ageMatch[1]);
                    const max = parseInt(ageMatch[2]);

                    // Показываем детектор возраста
                    if (ageDetection) {
                        ageDetection.classList.remove('hidden');
                        document.getElementById('min-age').textContent = min;
                        document.getElementById('max-age').textContent = max;
                    }

                    // Обновляем preview
                    if (previewAge) {
                        previewAge.textContent = `${min}-${max} лет`;
                    }

                    // Обновляем визуализацию
                    if (ageRangeVisual && ageRangeText) {
                        const percentage = (max / 30) * 100;
                        ageRangeVisual.style.width = `${Math.min(percentage, 100)}%`;
                        ageRangeText.textContent = `${min}-${max} лет`;
                    }

                    // Заполняем отдельные поля если они есть
                    if (minAgeInput && maxAgeInput) {
                        minAgeInput.value = min;
                        maxAgeInput.value = max;
                        updateAgeVisualization();
                    }
                } else {
                    if (ageDetection) {
                        ageDetection.classList.add('hidden');
                    }
                    if (previewAge) {
                        previewAge.textContent = 'не указан';
                    }
                }
            }

            if (descriptionInput && previewDescription) {
                previewDescription.textContent = descriptionInput.value || 'Описание появится здесь...';
                if (descriptionInput.value.length > 80) {
                    previewDescription.textContent = descriptionInput.value.substring(0, 80) + '...';
                }
            }
        }

        // Обновление визуализации возраста
        function updateAgeVisualization() {
            if (minAgeInput && maxAgeInput && ageRangeVisual && ageRangeText) {
                const min = parseInt(minAgeInput.value) || 0;
                const max = parseInt(maxAgeInput.value) || 0;

                if (min > 0 || max > 0) {
                    const percentage = (max / 30) * 100;
                    ageRangeVisual.style.width = `${Math.min(percentage, 100)}%`;
                    ageRangeText.textContent = `${min}-${max} лет`;

                    if (previewAge) {
                        previewAge.textContent = `${min}-${max} лет`;
                    }
                }
            }
        }

        // Обработчики событий
        if (nameInput) {
            nameInput.addEventListener('input', updatePreview);
        }

        if (descriptionInput) {
            descriptionInput.addEventListener('input', updatePreview);
        }

        if (minAgeInput) {
            minAgeInput.addEventListener('input', updateAgeVisualization);
        }

        if (maxAgeInput) {
            maxAgeInput.addEventListener('input', updateAgeVisualization);
        }

        // Выбор иконки
        const iconButtons = document.querySelectorAll('.icon-select');
        iconButtons.forEach(button => {
            button.addEventListener('click', function() {
                // Сброс предыдущего выбора
                iconButtons.forEach(btn => btn.classList.remove('selected'));

                // Установка нового выбора
                this.classList.add('selected');
                const icon = this.getAttribute('data-icon');

                if (iconInput) {
                    iconInput.value = icon;
                }

                if (previewIcon) {
                    previewIcon.textContent = icon;
                }
            });
        });

        // Примеры категорий
        const exampleButtons = document.querySelectorAll('button[data-example]');
        exampleButtons.forEach(button => {
            button.addEventListener('click', function() {
                const exampleName = this.getAttribute('data-example');
                const exampleText = this.querySelector('.font-medium').textContent;
                const exampleDesc = this.querySelector('.text-blue-600').textContent;

                if (nameInput) {
                    nameInput.value = exampleName;
                    nameInput.focus();
                }

                if (descriptionInput) {
                    descriptionInput.value = exampleDesc;
                }

                // Автовыбор подходящей иконки
                let targetIcon = '👶';
                if (exampleName.includes('Юниоры')) targetIcon = '👦';
                if (exampleName.includes('Взрослые')) targetIcon = '👨';
                if (exampleName.includes('10-12')) targetIcon = '👧';
                if (exampleName.includes('16-18')) targetIcon = '🧒';

                iconButtons.forEach(btn => {
                    if (btn.getAttribute('data-icon') === targetIcon) {
                        btn.click();
                    }
                });

                updatePreview();

                // Плавная прокрутка к форме
                document.querySelector('form').scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            });
        });



        // Инициализация предварительного просмотра
        updatePreview();

        // Автовыбор первой иконки
        if (iconButtons.length > 0 && !iconInput?.value) {
            iconButtons[0].click();
        }
    });
</script>