<?php

namespace app\controllers;

use app\models\ApplicationSearch;
use app\models\ExpertAssignment;
use app\models\Notification;
use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use app\models\Contest;
use app\models\Nomination;
use app\models\AgeCategory;
use app\models\User;
use app\models\Application;
use app\models\Evaluation;
use yii\helpers\ArrayHelper;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;


class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['index', 'contests', 'contest-create', 'contest-update', 'contest-delete', 'applications', 'application-view', 'users', 'user-create', 'user-update', 'user-delete', 'download-program', 'download-evaluations'],
                'rules' => [['allow' => true, 'roles' => ['@']]],
            ],
        ];
    }

    // Главная админка
    public function actionIndex()
    {
        return $this->render('index');
    }

    // ------------------- CRUD конкурсов -------------------
    public function actionContests()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Contest::find(),
            'pagination' => ['pageSize' => 20]
        ]);
        return $this->render('contests/index', ['dataProvider' => $dataProvider]);
    }

    public function actionContestCreate()
    {
        $model = new Contest();
        $nominations = Nomination::find()->all();
        $categories = AgeCategory::find()->all();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {

            // Сохраняем связи many-to-many
            $model->unlinkAll('nominations', true);
            if ($postNominations = Yii::$app->request->post('nominations', [])) {
                foreach ($postNominations as $nid) {
                    $model->link('nominations', Nomination::findOne($nid));
                }
            }

            $model->unlinkAll('ageCategories', true);
            if ($postCategories = Yii::$app->request->post('categories', [])) {
                foreach ($postCategories as $cid) {
                    $model->link('ageCategories', AgeCategory::findOne($cid));
                }
            }

            Yii::$app->session->setFlash('success', 'Конкурс создан');
            return $this->redirect(['contests']);
        }

        return $this->render('contests/create', [
            'model' => $model,
            'nominations' => $nominations,
            'categories' => $categories,
        ]);
    }

    public function actionContestUpdate($id)
    {
        $model = Contest::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException();

        $nominations = Nomination::find()->all();
        $categories = AgeCategory::find()->all();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {

            // Обновляем связи
            $model->unlinkAll('nominations', true);
            if ($postNominations = Yii::$app->request->post('nominations', [])) {
                foreach ($postNominations as $nid) {
                    $model->link('nominations', Nomination::findOne($nid));
                }
            }

            $model->unlinkAll('ageCategories', true);
            if ($postCategories = Yii::$app->request->post('categories', [])) {
                foreach ($postCategories as $cid) {
                    $model->link('ageCategories', AgeCategory::findOne($cid));
                }
            }

            Yii::$app->session->setFlash('success', 'Конкурс обновлен');
            return $this->redirect(['contests']);
        }

        return $this->render('contests/update', [
            'model' => $model,
            'nominations' => $nominations,
            'categories' => $categories,
        ]);
    }


    public function actionContestDelete($id)
    {
        $model = Contest::findOne($id);
        if ($model) $model->delete();
        return $this->redirect(['contests']);
    }

    // ------------------- CRUD пользователей -------------------
    public function actionBlock($id)
    {
        $user = User::findOne($id);

        if ($user === null) {
            throw new NotFoundHttpException('Пользователь не найден.');
        }

        if ($user->status !== 'blocked') {
            $user->status = 'blocked';  // Меняем статус на 'blocked'
            $user->save();  // Сохраняем изменения
            Yii::$app->session->setFlash('success', 'Пользователь заблокирован.');
        } else {
            Yii::$app->session->setFlash('warning', 'Пользователь уже заблокирован.');
        }

        return $this->redirect(['users']);  // Перенаправляем обратно на страницу пользователей
    }

    // Метод для разблокировки пользователя
    public function actionUnblock($id)
    {
        $user = User::findOne($id);

        if ($user === null) {
            throw new NotFoundHttpException('Пользователь не найден.');
        }

        if ($user->status !== 'active') {
            $user->status = 'active';  // Меняем статус на 'active'
            $user->save();  // Сохраняем изменения
            Yii::$app->session->setFlash('success', 'Пользователь разблокирован.');
        } else {
            Yii::$app->session->setFlash('warning', 'Пользователь уже активен.');
        }

        return $this->redirect(['users']);  // Перенаправляем обратно на страницу пользователей
    }

    // Метод для вывода списка пользователей
    public function actionUsers()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => User::find(),
            'pagination' => ['pageSize' => 20],
        ]);

        return $this->render('users/index', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionUserCreate()
    {
        $model = new User();  // Создаем новый объект модели User

        // Если данные успешно переданы и модель прошла валидацию
        if ($model->load(Yii::$app->request->post()) && $model->validate()) {
            // Хэшируем пароль перед сохранением
            $model->setPassword($model->password_hash); // предполагается, что password_hash это вводимый пароль

            // Устанавливаем статус активного пользователя
            $model->status = User::STATUS_ACTIVE;  // Используем константу STATUS_ACTIVE

            // Устанавливаем текущую дату и время в правильном формате
            $model->created_at = date('Y-m-d H:i:s');
            $model->updated_at = date('Y-m-d H:i:s');

            // Сохраняем пользователя в базу данных
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Пользователь успешно создан');
                return $this->redirect(['index']);  // Перенаправление на страницу со списком пользователей
            } else {
                Yii::$app->session->setFlash('error', 'Ошибка при создании пользователя');
            }
        }

        return $this->render('users/create', [
            'model' => $model,  // Передаем модель в представление
        ]);
    }


    public function actionUserUpdate($id)
    {
        // Находим модель пользователя по ID
        $model = User::findOne($id);

        // Если пользователь не найден, выбрасываем исключение
        if (!$model) {
            throw new \yii\web\NotFoundHttpException('Пользователь не найден');
        }

        // Загружаем данные из POST запроса и проверяем их
        if ($model->load(Yii::$app->request->post())) {

            // Проверяем, был ли введен новый пароль
            if (!empty($model->password_hash)) {
                // Если пароль изменен, хэшируем его перед сохранением
                $model->setPassword($model->password_hash); // Этот метод будет хэшировать пароль
            }

            // Сохраняем изменения модели
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Пользователь успешно обновлен');
                return $this->redirect(['users']);  // Перенаправляем на список пользователей
            } else {
                Yii::$app->session->setFlash('error', 'Ошибка при обновлении пользователя');
            }
        }

        // Если данные не прошли валидацию или форма не была отправлена, отображаем форму редактирования
        return $this->render('users/update', ['model' => $model]);
    }

    public function actionUserDelete($id)
    {
        $model = User::findOne($id);
        if ($model) $model->delete();
        return $this->redirect(['users']);
    }

    // ------------------- CRUD заявок -------------------

    public function actionApplications()
    {
        $searchModel = new ApplicationSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('applications/index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }



    public function actionApplicationView($id)
    {
        $model = $this->findApplication($id);

        return $this->render('applications/view', [
            'model' => $model,
        ]);
    }


    public function actionChangeStatus($id, $status)
    {
        $model = $this->findApplication($id);

        $model->status = $status;
        if ($model->save()) {
            Yii::$app->session->setFlash('success', 'Статус обновлен.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка сохранения статуса.');
        }

        return $this->redirect(['/admin/application-view', 'id' => $id]);
    }


    public function actionBlockApplication($id)
    {
        $model = $this->findApplication($id);
        $model->status = 'blocked';

        if ($model->save(false)) { // отключаем валидацию
            Yii::$app->session->setFlash('success', 'Заявка заблокирована.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка блокировки.');
        }

        return $this->redirect(['/admin/applications', 'id' => $id]);
    }



// Блокировка заявки
    public function actionUnblockApplication($id)
    {
        $model = $this->findApplication($id);

        $model->status = 'accepted'; // или 'new'

        if ($model->save(false)) {
            Yii::$app->session->setFlash('success', 'Заявка разблокирована.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка разблокировки.');
        }

        return $this->redirect(['/admin/applications', 'id' => $id]);
    }



    // Метод для поиска заявки по ID

    protected function findApplication($id)
    {
        if (($model = Application::findOne($id)) !== null) {
            return $model;
        }
        throw new NotFoundHttpException('Заявка не найдена.');
    }

    public function actionDeleteApplication($id)
    {
        $model = $this->findApplication($id);

        if ($model->delete()) {
            Yii::$app->session->setFlash('success', 'Заявка удалена.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка удаления.');
        }

        return $this->redirect(['applications']);
    }

    public function actionEditApplication($id)
    {
        $model = $this->findApplication($id);

        if ($model->load(Yii::$app->request->post())) {
            $model->save(false); // игнорируем валидацию
            Yii::$app->session->setFlash('success', 'Заявка обновлена.');
            return $this->redirect(['applications', 'id' => $model->id]);
        }


        return $this->render('applications/edit', [
            'model' => $model,
        ]);
    }




    // ------------------- Отчеты -------------------
    public function actionDownloadProgram($contest_id)
    {
        $contest = Contest::findOne($contest_id);
        if (!$contest) throw new \yii\web\NotFoundHttpException();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setCellValue('A1', 'Номер');
        $sheet->setCellValue('B1', 'Номинация');
        $sheet->setCellValue('C1', 'ФИО');
        $sheet->setCellValue('D1', 'Название номера');

        $row = 2;
        foreach ($contest->applications as $i => $app) {
            $sheet->setCellValue('A' . $row, $i + 1);
            $sheet->setCellValue('B' . $row, $app->nomination->name);
            $sheet->setCellValue('C' . $row, $app->user->surname . ' ' . $app->user->name . ' ' . $app->user->patronymic);
            $sheet->setCellValue('D' . $row, $app->title);
            $row++;
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'program_' . $contest->id . '.xlsx';
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=$filename");
        $writer->save("php://output");
        exit;
    }

    public function actionDownloadEvaluations($contest_id)
    {
        $contest = Contest::findOne($contest_id);
        if (!$contest) throw new \yii\web\NotFoundHttpException();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['Номер', 'ФИО / Название коллектива', 'Руководитель', 'Название номера и автор', 'Мастерство', 'Артистизм', 'Сценическая культура', 'Общий балл', 'Примечания'];
        foreach ($headers as $i => $h) $sheet->setCellValue(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($i + 1) . '1', $h);


        $row = 2;
        foreach ($contest->applications as $i => $app) {
            $evaluations = $app->evaluations;
            $mastery = 0;
            $artistry = 0;
            $stage_culture = 0;
            foreach ($evaluations as $ev) {
                $mastery += $ev->mastery;
                $artistry += $ev->artistry;
                $stage_culture += $ev->stage_culture;
            }
            $count = max(count($evaluations), 1);
            $sheet->setCellValue('A' . $row, $i + 1);
            $sheet->setCellValue('B' . $row, $app->participant_surname . ' ' . $app->participant_name . ' ' . $app->participant_patronymic);
            $sheet->setCellValue('C' . $row, $app->supervisor);
            $sheet->setCellValue('D' . $row, $app->title);
            $sheet->setCellValue('E' . $row, round($mastery / $count, 2));
            $sheet->setCellValue('F' . $row, round($artistry / $count, 2));
            $sheet->setCellValue('G' . $row, round($stage_culture / $count, 2));
            $sheet->setCellValue('H' . $row, round(($mastery + $artistry + $stage_culture) / $count, 2));
            $sheet->setCellValue('I' . $row, ''); // Примечания
            $row++;
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'evaluations_' . $contest->id . '.xlsx';
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=$filename");
        $writer->save("php://output");
        exit;
    }

    public function actionAssignExpert()
    {
        $model = new ExpertAssignment();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {

            $expertId = $model->expert_id;

            // достаем ВСЕ заявки конкурса
            $applications = Application::find()
                ->where(['contest_id' => $model->contest_id])
                ->all();

            // создаём оценки эксперту
            foreach ($applications as $app) {

                // если уже есть — пропускаем
                if (!Evaluation::find()->where([
                    'application_id' => $app->id,
                    'expert_id' => $expertId
                ])->exists()) {

                    $eval = new Evaluation();
                    $eval->application_id = $app->id;
                    $eval->expert_id = $expertId;
                    $eval->status = 'draft';
                    $eval->created_at = date('Y-m-d H:i:s');
                    $eval->updated_at = date('Y-m-d H:i:s');
                    $eval->save(false);
                }
            }

            // уведомление эксперту
            $expert = User::findOne($expertId);
            $contest = Contest::findOne($model->contest_id);

            if ($expert && $contest) {
                $notification = new Notification();
                $notification->user_id = $expert->id;
                $notification->title = 'Новое назначение';
                $notification->message = 'Вам назначены заявки на конкурс: ' . $contest->title;
                $notification->is_read = 0;
                $notification->created_at = date('Y-m-d H:i:s');
                $notification->save();
            }

            Yii::$app->session->setFlash('success', 'Эксперт назначен');
            return $this->redirect(['expert-assignments']);
        }

        return $this->render('expert-assign/assign', [
            'model' => $model,
            'experts' => User::find()->where(['role' => 'expert'])->all(),
            'contests' => Contest::find()->all(),
            'nominations' => Nomination::find()->all(),
            'categories' => AgeCategory::find()->all(),
        ]);
    }



    public function actionExpertMyApplications()
    {
        $expertId = Yii::$app->user->id;

        // назначения эксперта
        $assignments = ExpertAssignment::find()
            ->where(['expert_id' => $expertId])
            ->all();

        if (!$assignments) {
            return $this->render('expert/my-applications', [
                'applications' => [],
            ]);
        }

        $query = Application::find()->alias('a');

        // добавляем условия для КАЖДОГО назначения
        $query->where(['OR']);

        foreach ($assignments as $as) {

            $cond = ['a.contest_id' => $as->contest_id];

            if ($as->nomination_id !== null) {
                $cond['a.nomination_id'] = $as->nomination_id;
            }

            if ($as->age_category_id !== null) {
                $cond['a.age_category_id'] = $as->age_category_id;
            }

            // добавляем условие
            $query->orWhere($cond);
        }

        $applications = $query->all();

        return $this->render('expert/my-applications', [
            'applications' => $applications,
        ]);
    }


    public function actionDeleteFiles($contest_id)
    {
        $applications = Application::find()->where(['contest_id' => $contest_id])->all();
        foreach ($applications as $app) {
            if (file_exists($app->file_path)) unlink($app->file_path);
            $app->file_path = null;
            $app->save();
        }
        Yii::$app->session->setFlash('success', 'Файлы удалены');
        return $this->redirect(['contests']);
    }



// ------------------- CRUD возрастных категорий -------------------
    public function actionAgeCategories()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => AgeCategory::find(),
            'pagination' => ['pageSize' => 20]
        ]);
        return $this->render('age-category/index', ['dataProvider' => $dataProvider]);
    }

    public function actionAgeCategoryCreate()
    {
        $model = new AgeCategory();
        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Возрастная категория создана');
            return $this->redirect(['age-categories']);
        }
        return $this->render('age-category/create', ['model' => $model]);
    }

    public function actionAgeCategoryUpdate($id)
    {
        $model = AgeCategory::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException();
        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Возрастная категория обновлена');
            return $this->redirect(['age-categories']);
        }
        return $this->render('age-category/update', ['model' => $model]);
    }

    public function actionAgeCategoryDelete($id)
    {
        $model = AgeCategory::findOne($id);
        if ($model) $model->delete();
        Yii::$app->session->setFlash('success', 'Возрастная категория удалена');
        return $this->redirect(['age-categories']);
    }

// ------------------- CRUD номинаций -------------------
    public function actionNominations()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Nomination::find(),
            'pagination' => ['pageSize' => 20]
        ]);
        return $this->render('nomination/index', ['dataProvider' => $dataProvider]);
    }

    public function actionNominationCreate()
    {
        $model = new Nomination();
        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Номинация создана');
            return $this->redirect(['nominations']);
        }
        return $this->render('nomination/create', ['model' => $model]);
    }

    public function actionNominationUpdate($id)
    {
        $model = Nomination::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException();
        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Номинация обновлена');
            return $this->redirect(['nominations']);
        }
        return $this->render('nomination/update', ['model' => $model]);
    }

    public function actionNominationDelete($id)
    {
        $model = Nomination::findOne($id);
        if ($model) $model->delete();
        Yii::$app->session->setFlash('success', 'Номинация удалена');
        return $this->redirect(['nominations']);
    }

// ------------------- Назначения экспертов -------------------
    public function actionExpertAssignments()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => ExpertAssignment::find()->with(['expert', 'contest', 'nomination', 'ageCategory']),
            'pagination' => ['pageSize' => 20]
        ]);

        return $this->render('expert-assign/index', ['dataProvider' => $dataProvider]);
    }


    public function actionEvaluations($applicationId = null)
    {
        $query = Evaluation::find()->joinWith(['application', 'expert']);

        if ($applicationId) {
            $query->andWhere(['application_id' => $applicationId]);
        }

        $dataProvider = new \yii\data\ActiveDataProvider([
            'query' => $query,
            'pagination' => ['pageSize' => 20],
            'sort' => ['defaultOrder' => ['id' => SORT_DESC]],
        ]);

        return $this->render('evaluations/index', [
            'dataProvider' => $dataProvider,
        ]);
    }
    public function actionContestView($id)
    {
        $model = Contest::findOne($id);
        if (!$model) throw new \yii\web\NotFoundHttpException();

        return $this->render('contests/view', [
            'model' => $model,
        ]);
    }

    public function actionResetEvaluation($id)
    {
        $model = Evaluation::findOne($id);
        if (!$model) {
            throw new \yii\web\NotFoundHttpException('Оценка не найдена.');
        }

        $model->status = 'draft'; // Сбрасываем на "Ожидает оценки"
        if ($model->save()) {
            Yii::$app->session->setFlash('success', 'Статус оценки сброшен.');
        } else {
            Yii::$app->session->setFlash('error', 'Ошибка при сбросе статуса.');
        }

        return $this->redirect(['evaluations', 'applicationId' => $model->application_id]);
    }

    public function actionNotifications()
    {
        $dataProvider = new \yii\data\ActiveDataProvider([
            'query' => Notification::find()->with('user')->orderBy(['id' => SORT_DESC]),
            'pagination' => ['pageSize' => 20],
        ]);

        return $this->render('notifications/index', [
            'dataProvider' => $dataProvider
        ]);
    }

    public function actionCreateNotification()
    {
        $model = new Notification();
        $users = User::find()->all();

        if ($model->load(Yii::$app->request->post())) {

            $target = Yii::$app->request->post('target');

            // 🔹 Рассылка всем пользователям
            if ($target === 'all') {
                foreach ($users as $u) {
                    $n = new Notification();
                    $n->user_id = $u->id;
                    $n->title = $model->title;
                    $n->message = $model->message;
                    $n->is_read = 0;
                    $n->save();
                }
            }

            // 🔹 Только экспертам
            elseif ($target === 'experts') {
                $experts = User::find()->where(['role' => 'expert'])->all();
                foreach ($experts as $u) {
                    $n = new Notification();
                    $n->user_id = $u->id;
                    $n->title = $model->title;
                    $n->message = $model->message;
                    $n->is_read = 0;
                    $n->save();
                }
            }

            // 🔹 Только участникам
            elseif ($target === 'users') {
                $simpleUsers = User::find()->where(['role' => 'user'])->all();
                foreach ($simpleUsers as $u) {
                    $n = new Notification();
                    $n->user_id = $u->id;
                    $n->title = $model->title;
                    $n->message = $model->message;
                    $n->is_read = 0;
                    $n->save();
                }
            }

            // 🔹 Конкретному пользователю
            else {
                $model->is_read = 0;
                $model->save();
            }

            Yii::$app->session->setFlash('success', 'Уведомления отправлены.');
            return $this->redirect(['notifications']);
        }

        return $this->render('notifications/create', [
            'model' => $model,
            'users' => $users
        ]);
    }

}
